/*
    Calibration.cpp
    Author: Seb Madgwick
*/

//------------------------------------------------------------------------------
// Includes

#include "Calibration.h"
#include <EEPROM.h>
#include "generic.h"
#include "ADXL345.h"
#include "HMC5883L.h"
#include "L3G4200D.h"


//------------------------------------------------------------------------------
// Definitions

typedef union {
    int intVal;
    struct {
        char lsb;
        char msb;
    };
} IntUnion;

//------------------------------------------------------------------------------
// Variables

// Public
int Calibration::reg[NUM_REGISTERS];                            // calibration registers
float Calibration::gyrX, Calibration::gyrY, Calibration::gyrZ;  // calibrated gyroscope data (0.1 dps)
float Calibration::accX, Calibration::accY, Calibration::accZ;  // calibrated accelerometer data (mg)
float Calibration::magX, Calibration::magY, Calibration::magZ;  // calibrated magnetometer data (mGa)

// Private
float Calibration::grx, Calibration::gry, Calibration::grz; // gyroscope sensitivity reciprocal (0.1 dps/lsb)
float Calibration::gbx, Calibration::gby, Calibration::gbz; // gyroscope bias (lsb)
float Calibration::arx, Calibration::ary, Calibration::arz; // accelerometer sensitivity reciprocal (mg/lsb)
float Calibration::abx, Calibration::aby, Calibration::abz; // accelerometer bias (lsb)
float Calibration::sxx, Calibration::sxy, Calibration::sxz; // soft-iron x-axis parameters
float Calibration::syx, Calibration::syy, Calibration::syz; // soft-iron y-axis parameters
float Calibration::szx, Calibration::szy, Calibration::szz; // soft-iron z-axis parameters
float Calibration::hx, Calibration::hy, Calibration::hz;    // hard-iron parameters (mGa)

//------------------------------------------------------------------------------
// Methods

void Calibration::init() {

    // Load all registers from EEPROM
    for(int address = 0; address < NUM_REGISTERS; address++) {
        IntUnion intUnion;
        intUnion.lsb = EEPROM.read(2 * address);
        intUnion.msb = EEPROM.read((2 * address) + 1);
        reg[address] = intUnion.intVal;
    }

    // Check if registers are blank
    if(reg[REG_IS_BLANK] != 0) {

        // Load defaults
        // L3G4200D
        reg[REG_IS_BLANK]       = 0;      
        reg[REG_GYR_SENS_X]     = 875;    // 0.00875 lsb/dps  //EEPROM 1
        reg[REG_GYR_SENS_Y]     = 875;
        reg[REG_GYR_SENS_Z]     = 875;
        reg[REG_GYR_BIAS_X]     = 0;        // 0.1 lsb       //EEPROM 4
        reg[REG_GYR_BIAS_Y]     = 0;
        reg[REG_GYR_BIAS_Z]     = 0;
        // ADXL345
        reg[REG_ACC_SENS_X]     = 3926;    // 0.1 lsb/g @ +-16G range    // EEPROM 7
        reg[REG_ACC_SENS_Y]     = 3926;
        reg[REG_ACC_SENS_Z]     = 3926;
        reg[REG_ACC_BIAS_X]     = 0;        // 0.1 lsb      //EEPROM 10
        reg[REG_ACC_BIAS_Y]     = 0;
        reg[REG_ACC_BIAS_Z]     = 0;
        
        reg[REG_MAG_SOFT_XX]    = 10900;    // 10000       // EEPROM 13
        reg[REG_MAG_SOFT_XY]    = 0;
        reg[REG_MAG_SOFT_XZ]    = 0;
        reg[REG_MAG_SOFT_YX]    = 0;
        reg[REG_MAG_SOFT_YY]    = 10900;   // 10000;       // EEPROM 17
        reg[REG_MAG_SOFT_YZ]    = 0;
        reg[REG_MAG_SOFT_ZX]    = 0;
        reg[REG_MAG_SOFT_ZY]    = 0;
        reg[REG_MAG_SOFT_ZZ]    = 10900;  // 10000;       // EEPROM 21
        
        reg[REG_MAG_HARD_X]     = 0;        // 0.1 mGa  // EEPROM 22
        reg[REG_MAG_HARD_Y]     = 0;
        reg[REG_MAG_HARD_Z]     = 0;

        // Save defaults to EEPROM
        saveToEEPROM();
    }
    createCalibrationParams();
}

void Calibration::createCalibrationParams() {

    // Create calibration parameters
    grx = (float)reg[REG_GYR_SENS_X] / 10000.0f;
    gry = (float)reg[REG_GYR_SENS_Y] / 10000.0f;
    grz = (float)reg[REG_GYR_SENS_Z] / 10000.0f;
    gbx =  (float)reg[REG_GYR_BIAS_X] / 10.0f;
    gby =  (float)reg[REG_GYR_BIAS_Y] / 10.0f;
    gbz =  (float)reg[REG_GYR_BIAS_Z] / 10.0f;
    
    arx = (float)reg[REG_ACC_SENS_X] / 10000.0f;
    ary = (float)reg[REG_ACC_SENS_Y] / 10000.0f;
    arz = (float)reg[REG_ACC_SENS_Z] / 10000.0f;
    abx =  (float)reg[REG_ACC_BIAS_X] / 10.0f;
    aby =  (float)reg[REG_ACC_BIAS_Y] / 10.0f;
    abz =  (float)reg[REG_ACC_BIAS_Z] / 10.0f;
    
    sxx = (1.0f / 10000.0f) * (float)reg[REG_MAG_SOFT_XX];
    sxy = (1.0f / 10000.0f) * (float)reg[REG_MAG_SOFT_XY];
    sxz = (1.0f / 10000.0f) * (float)reg[REG_MAG_SOFT_XZ];
    syx = (1.0f / 10000.0f) * (float)reg[REG_MAG_SOFT_YX];
    syy = (1.0f / 10000.0f) * (float)reg[REG_MAG_SOFT_YY];
    syz = (1.0f / 10000.0f) * (float)reg[REG_MAG_SOFT_YZ];
    szx = (1.0f / 10000.0f) * (float)reg[REG_MAG_SOFT_ZX];
    szy = (1.0f / 10000.0f) * (float)reg[REG_MAG_SOFT_ZY];
    szz = (1.0f / 10000.0f) * (float)reg[REG_MAG_SOFT_ZZ];
    
    hx = 0.1f * (float)reg[REG_MAG_HARD_X];
    hy = 0.1f * (float)reg[REG_MAG_HARD_Y];
    hz = 0.1f * (float)reg[REG_MAG_HARD_Z];
}

void Calibration::update() {
    createCalibrationParams();
    xyzInts_t* gyro = readGyroXYZ();
    gyrX = grx * (gyro->x - gbx);
    gyrY = gry * (gyro->y - gby);
    gyrZ = grz * (gyro->z - gbz);
    
    xyzInts_t* accel = readAccelXYZ();
    accX = arx * (accel->x - abx);
    accY = ary * (accel->y - aby);
    accZ = arz * (accel->z - abz);
    
    xyzInts_t* magneto = readMagnetoXYZ();
    float tmpMagx = (1000.0f / 390.0f) * magneto->x;
    float tmpMagy = (1000.0f / 390.0f) * magneto->y;
    float tmpMagz = (1000.0f / 390.0f) * magneto->z;
    magX = sxx * tmpMagx + sxy * tmpMagy + sxz * tmpMagz - hx;
    magY = syx * tmpMagx + syy * tmpMagy + syz * tmpMagz - hy;
    magZ = szx * tmpMagx + szy * tmpMagy + szz * tmpMagz - hz;
}

void Calibration::saveToEEPROM() {
    for(int address = 0; address < NUM_REGISTERS; address++) {
        IntUnion intUnion;
        intUnion.intVal = reg[address];
        EEPROM.write(2 * address, intUnion.lsb);
        EEPROM.write((2 * address) + 1, intUnion.msb);
    }
}

//------------------------------------------------------------------------------
// End of file
