/*
Version 5
*/

#include <OscSerial.h>
#include <EthernetUdp.h>
#include <SPI.h>  
#include <Wire.h>

#include "generic.h"
#include "I2C.h"
#include "L3G4200D.h"
#include "HMC5883L.h"
#include "ADXL345.h"
#include "AHRS.h"
#include "Calibration.h"
#include <EEPROM.h> // required by Calibration.cpp

#define LED 13

OscSerial oscSerial;
AHRS ahrs;

long timer;
int rate = 100; // sample rate in ms.
float sampleFrequency = 1000 / (float)rate;


void setup(){
    pinMode(LED, OUTPUT);
    Wire.begin();
    Serial.begin(57600);
    oscSerial.begin(Serial);
    delay(1000);  

//------  L3G4200D Gyro 
    while (!L3G4200D_init(L3G4200D_SCALE_2K)) {
        digitalWrite(LED, !digitalRead(LED));
        localOscDebug("L3G4200D not found");
        delay(500);
    }
 
//------  ADXL345 Accelerometer  
    while(!ADXL345_init(ADXL345_RANGE_2_G)) {
        digitalWrite(LED, !digitalRead(LED));
        localOscDebug("ADXL345 not found");
        delay(500);
    }
    
//------  HMC5883 compass
    while (!HMC5883L_init(HMC5883L_GAIN_1P3G)) {
        digitalWrite(LED, !digitalRead(LED));
        localOscDebug("HMC5883L not found");
        delay(500);
    }
    
//------  BMP085 Pressure & Temp

 //   BMP085_bmp.begin();
  
//------  AHRS
    ahrs.init(sampleFrequency);
  
  
//------  Calibration
    Calibration::init();
    
//------  Startup message  
    delay(1000); //wait for all the sensors to be ready 
 
    OscMessage m;
    sendFwVersion(m);   
//    localOscDebug("starting");
}

void loop(){
  long now = millis();
  if (now-timer > rate) {  
    Calibration::update();

    
//------  L3G4200D Gyro 
//    xyzInts_t* gyro = readGyroXYZ();	

    OscMessage msgG("/11AoC/gyro");
    msgG.add(Calibration::gyrX);
    msgG.add(Calibration::gyrY);
    msgG.add(Calibration::gyrZ);
    msgG.add(getGyroTemp());
    oscSerial.send(msgG); 

//------  ADXL345 Accelerometer 

//    xyzInts_t* accel = readAccelXYZ();
 
    /* Display the results (acceleration is measured in m/s^2) */
    OscMessage msgA("/11AoC/accel"); 
    msgA.add(Calibration::accX); 
    msgA.add(Calibration::accY); 
    msgA.add(Calibration::accZ); 
    oscSerial.send(msgA); 

//------  HMC5883 compass
    xyzInts_t* magneto = readMagnetoXYZ();
  
    // Print raw values
    OscMessage msgC("/11AoC/compass");
    msgC.add(Calibration::magX);
    msgC.add(Calibration::magY);
    msgC.add(Calibration::magZ);
    oscSerial.send(msgC);
  
//------  BMP085 Pressure & Temp 
//    float BMP085_temp = BMP085_bmp.readTemperature();
//    long  BMP085_pressure = BMP085_bmp.readPressure();
  
//    OscMessage msgB("/11AoC/barotemp");
//    msgB.add(BMP085_temp);
//    msgB.add(BMP085_pressure);
//    oscSerial.send(msgB);
    
  // Update AHRS algorithm
//    ahrs.update(DegToRad(0.1f * Calibration::gyrX), DegToRad(0.1f * Calibration::gyrY), DegToRad(0.1f * Calibration::gyrZ), 
//                         Calibration::accX , Calibration::accY, Calibration::accZ);
//    ahrs.update2(DegToRad(0.1f * Calibration::gyrX), DegToRad(0.1f * Calibration::gyrY), DegToRad(0.1f * Calibration::gyrZ), 
//                         Calibration::accX , Calibration::accY, Calibration::accZ);
    ahrs.updateAHRS(DegToRad(0.1f * Calibration::gyrX), DegToRad(0.1f * Calibration::gyrY), DegToRad(0.1f * Calibration::gyrZ), 
                         Calibration::accX , Calibration::accY, Calibration::accZ,
                         Calibration::magX , Calibration::magY, Calibration::magZ);
    
    OscMessage msgQ("/11AoC/quats");
    msgQ.add(ahrs.q0);
    msgQ.add(ahrs.q1);
    msgQ.add(ahrs.q2);    
    msgQ.add(ahrs.q3);
    oscSerial.send(msgQ);
    
    timer = now;
  }
    
  oscSerial.listen();
}

void localOscDebug(char *message) {
    OscMessage msgB("/11AoC/debug");
    msgB.add(message);
    oscSerial.send(msgB);   
}

// OSC incoming message dispatcher
void oscEvent(OscMessage &m) { // *note the & before msg
    // receive a message 
    m.plug("/11AoC/LED",          ledSwitch);
    m.plug("/11AoC/version",      sendFwVersion);
    m.plug("/11AoC/refreshRate",  setRefreshRate);
    m.plug("/11AoC/readEEPROM",   readEEPROM);
    m.plug("/11AoC/saveCalibration", saveCalibration);
    m.plug("/11AoC/accelBias",    accelBias);
    m.plug("/11AoC/gyroBias",     gyroBias);
    m.plug("/11AoC/magBias",      magBias);
    m.plug("/11AoC/magSens",      magSensitivity);
    m.plug("/11AoC/accelSens",    accelSensitivity);
    m.plug("/11AoC/gyroSens",     gyroSensitivity);
}

void ledSwitch(OscMessage &m) {  // *note the & before msg
  // getting to the message data 
  int value = m.getInt(0); 
  if (value == 0) digitalWrite(13, LOW);
  if (value == 1) digitalWrite(13, HIGH);
}

void sendFwVersion(OscMessage &m) {
    OscMessage msgV("/11AoC/version");
    msgV.add(5);
    oscSerial.send(msgV);  
}

void setRefreshRate(OscMessage &m) {
    rate = m.getInt(0);
    OscMessage msgR("/11AoC/refreshRate");
    msgR.add(rate);
    oscSerial.send(msgR);
}

void readEEPROM(OscMessage &m) {
    byte address = (byte)m.getInt(0);
    OscMessage msgE("/11AoC/EEPROM");
    msgE.add(address);
    msgE.add(((EEPROM.read(2 * address + 1)) << 8) + (EEPROM.read(2 * address)));
    oscSerial.send(msgE);
}

void accelBias(OscMessage &m) {
    Calibration::reg[REG_ACC_BIAS_X] = m.getInt(0);
    Calibration::reg[REG_ACC_BIAS_Y] = m.getInt(1);
    Calibration::reg[REG_ACC_BIAS_Z] = m.getInt(2);
}

void gyroBias(OscMessage &m) {
    Calibration::reg[REG_GYR_BIAS_X] = m.getInt(0);
    Calibration::reg[REG_GYR_BIAS_Y] = m.getInt(1);
    Calibration::reg[REG_GYR_BIAS_Z] = m.getInt(2);
}

void magBias(OscMessage &m) {
    Calibration::reg[REG_MAG_HARD_X] = m.getInt(0);
    Calibration::reg[REG_MAG_HARD_Y] = m.getInt(1);
    Calibration::reg[REG_MAG_HARD_Z] = m.getInt(2);
}

void accelSensitivity(OscMessage &m) {
    Calibration::reg[REG_ACC_SENS_X] = m.getInt(0);
    Calibration::reg[REG_ACC_SENS_Y] = m.getInt(1);
    Calibration::reg[REG_ACC_SENS_Z] = m.getInt(2);
}

void gyroSensitivity(OscMessage &m) {
    Calibration::reg[REG_GYR_SENS_X] = m.getInt(0);
    Calibration::reg[REG_GYR_SENS_Y] = m.getInt(1);
    Calibration::reg[REG_GYR_SENS_Z] = m.getInt(2);
}

void magSensitivity(OscMessage &m) {
    Calibration::reg[REG_MAG_SOFT_XX] = m.getInt(0);
    Calibration::reg[REG_MAG_SOFT_YY] = m.getInt(1);
    Calibration::reg[REG_MAG_SOFT_ZZ] = m.getInt(2);
}

void saveCalibration(OscMessage &m) {
    Calibration::saveToEEPROM();
    digitalWrite(LED, !digitalRead(LED));
    delay(100);
    digitalWrite(LED, !digitalRead(LED));
    delay(100);
    digitalWrite(LED, !digitalRead(LED));
    delay(100);
    digitalWrite(LED, !digitalRead(LED));
}
