;**********************************************************************
;**                    Midi Exiter Library 0.35                      **
;**********************************************************************
;
; This file contains the more or less stable subroutines of the 
; MIDI Exiter (midiexiter.asm). Files copyrighted by F.J. Kraan
; unless indicated otherwise. When all foreign routines are replaced,
; the set will be released under GPL.
; 
.cseg

; The delay routines are adapted from  (c)andreas-s@web.de
;  Never checked the values, just multiplied by two.
;Pause nach jeder bertragung
DELAY5US:                              ;50us Pause
           ldi  sdelcnt, $84
DELAY5US_: dec  sdelcnt
           brne DELAY5US_
           ret                          ;wieder zurck

;Lngere Pause fr manche Befehle
DELAY5MS:                               ;5ms Pause
           ldi  ldelcnt, $42
WGLOOP0:   ldi  sdelcnt, $C9
WGLOOP1:   dec  sdelcnt
           brne WGLOOP1
           dec  ldelcnt
           brne WGLOOP0
           ret                          ;wieder zurck


;**                    Keyboard routine
; Uses two phases, controlled by keyphfl. One phase only to get a key
;  value, and another to OR this value with a new value. The result is
;  the scanbut value. This is designed as a delay-less debounce. The 
;  tradeoff is a keyboard refresh once in two cycles.
; It still doesn't work very well.
READBUTTON:
    cbi   ButCol, NavCol   ; Enable navigation buttons
	sbi   ButCol, FncCol   ; Disable function buttons
    rcall DELAY5US
    in    temp,   ButRows  ; Read value
    andi  temp,   ButMask  ; Mask off non button bits
    mov   temp2,  temp     ; Store this part
    swap  temp2            ; Swap it to other nibble
    cbi   ButCol, FncCol   ; Enable function buttons
    sbi   ButCol, NavCol   ; Disable navigation buttons
    rcall DELAY5US
    in    temp,   ButRows  ; Read value
    andi  temp,   ButMask  ; Mask off non button bits
    or    temp2,  temp     ; Add it to stored value
	com   temp2
	sbrc  flags,  tstkeyphfl
	rjmp  NOKEYPHASE
KEYPHASE:
	mov   scanbut,temp2    ; 
	or    scanbut,tempscan
	cbr   flags,  setkeyphfl
	rjmp  READBUT_EXIT
NOKEYPHASE:
	mov   tempscan,scanbut
	sbr   flags,  setkeyphfl
READBUT_EXIT:
    sbi   ButCol, FncCol   ; Disable function buttons
    sbi   ButCol, NavCol   ; Disable navigation buttons
    ret
	mov   tempscan, scanbut ; a button toggle!

BUSYLED:
	sbic  RateLedPort,  RateLedBit      ; Skip next if bit is clear
    rjmp  BL_SET
BL_CLEAR:
	sbi   RateLedPort,  RateLedBit
	ret
BL_SET:
	cbi   RateLedPort,  RateLedBit
	ret

HD12_CLEAR:
	ldi   temp,    0b11110000
	out   HD_Data, temp
	ldi   temp,    HD_1LSN      ; Address bit pattern 0-2 LCD, 3-5 HD
	out   HD_Adr,  temp         ; Write to port
    rcall HD_WRPULSE            ; Pulse the address enable strobe
	ldi   temp,    HD_1MSN      ; Address bit pattern 0-2 LCD, 3-5 HD
	out   HD_Adr,  temp         ; Write to port
    rcall HD_WRPULSE            ; Pulse the address enable strobe
	ldi   temp,    HD_2LSN      ; Address bit pattern 0-2 LCD, 3-5 HD
	out   HD_Adr,  temp         ; Write to port
    rcall HD_WRPULSE            ; Pulse the address enable strobe
	ldi   temp,    HD_2MSN      ; Address bit pattern 0-2 LCD, 3-5 HD
	out   HD_Adr,  temp         ; Write to port
    rcall HD_WRPULSE            ; Pulse the address enable strobe
	ret

HD1_WR:                         ; MSB 7-segment display routine
	rcall HD_GETCHAR
	mov   temp2,   temp         ; Keep value for second part
	swap  temp
	out   HD_Data, temp
	ldi   temp,    HD_1LSN      ; 
	out   HD_Adr,  temp         ; Write to port
    rcall HD_WRPULSE            ; Pulse the address enable strobe
	out   HD_Data, temp2
	ldi   temp,    HD_1MSN      ; 
	out   HD_Adr,  temp         ; Write to port
    rcall HD_WRPULSE            ; Pulse the address enable strobe
	ret

HD2_WR:                         ; LSB 7-segment display routine
	rcall HD_GETCHAR
	mov   temp2,   temp         ; Keep value for second part
	swap  temp
	out   HD_Data, temp
	ldi   temp,    HD_2LSN      ; 
	out   HD_Adr,  temp         ; Write to port
    rcall HD_WRPULSE            ; Pulse the address enable strobe
	out   HD_Data, temp2
	ldi   temp,    HD_2MSN      ; 
	out   HD_Adr,  temp         ; Write to port
    rcall HD_WRPULSE            ; Pulse the address enable strobe
	ret

HD3_WR:
	swap  temp
	andi  temp,    0xF0
	out   HD_Data, temp
	ldi   temp,    HD_3         ; Address bit pattern 0-2 LCD, 3-5 HD
	out   HD_Adr,  temp         ; Write to port
    rcall HD_WRPULSE            ; Pulse the address enable strobe
	ret

HD_WRPULSE:
	cbi   PORTC,   4
	rcall DELAY5US
	sbi   PORTC,   4
	ret

HD_GETCHAR:
	andi  temp,    0x0F
	mov   ramptrl, temp
	ldi   temp,    LOW(HDCHAR)
	add   ramptrl, temp
	ld    temp,    X
	ret

EDIT_SELECT:; Set edit mode with < and > keys
    sbrc  scanbut, 4        ; > key
	inc   editstat
	sbrc  scanbut, 7        ; < key
	dec   editstat
    cpi   editstat,    0x05
	breq  ES_ZERO
	cpi   editstat,    0xFF
	brne  ES_OK
ES_FF:
	ldi   temp,        0x04
	mov   editstat,    temp
	rjmp  ES_OK
ES_ZERO:
	ldi   temp,        0x00
	mov   editstat,    temp
ES_OK:
	ret

EDIT_DISPLAY:           ;Display editable parameter value and related LED pattern
	cpi   editstat,    0
	breq  ED_WAVTYP
	cpi   editstat,    1
	breq  ED_WAVDS
	cpi   editstat,    2
	breq  ED_MSGRAT
	cpi   editstat,    3
	breq  ED_MCHAN
	cpi   editstat,    4
	breq  ED_MCTRN
	rjmp  ED_EXIT
ED_WAVTYP:                         ; 0
	cbr   flags2, setInhMsgfl
	rcall HD12_CLEAR
	mov   temp,  wavemode
	rcall HD2_WR
	ldi   temp,  0b11111110
	rcall HD3_WR
	rjmp  ED_EXIT
ED_WAVDS:                          ; 1
	cbr   flags2, setInhMsgfl
	mov   temp,  dutycycl
	swap  temp
	rcall HD1_WR
	mov   temp,  dutycycl
	rcall HD2_WR
	ldi   temp,  0b11111101
	rcall HD3_WR
	rjmp  ED_EXIT
ED_MSGRAT:                          ; 2
	cbr   flags2, setInhMsgfl
	mov   temp,  msgrate
	swap  temp
	rcall HD1_WR
	mov   temp,  msgrate
	rcall HD2_WR
	ldi   temp,  0b11111011
	rcall HD3_WR
	rjmp  ED_EXIT
ED_MCHAN:                          ; 3
	cbr   flags2, setInhMsgfl
	mov   temp,  midichan
	rcall HD12_CLEAR
	mov   temp,  midichan
	rcall HD2_WR
	ldi   temp,  0b11110111
	rcall HD3_WR
	rjmp  ED_EXIT
ED_MCTRN:                          ; 4
	sbr   flags2, setInhMsgfl
	mov   temp,  midictrn
	swap  temp
	rcall HD1_WR
	mov   temp,  midictrn
	rcall HD2_WR
	ldi   temp,  0b11111100
	rcall HD3_WR
ED_EXIT:
	ret
; 
; Ring buffer principles.
; Each ring buffer has the following parameters:
; RBufStr - Starting point of the buffer in memory. Fixed address.
; RBufEnd - End point of the buffer in memory. Fixed address.
; HeadPtr - Pointer to the header of the buffer contents. Here the 
;           ADD2?XRB (add to) routine adds characters. The HeadPtr
;           points to the first free address.
; TailPtr - Pointer to the tail of the buffer contents. Here the
;           GET4?XRB (get from) routine removes characters. The
;           TailPtr points to the character to be removed.
; ?xrbstat - The size of the buffer contents. Zero when empty.
;
; The checking is minimal, The calling program must check the buffer
; status before calling the GET4?XRB. Overflows should also be detected
; by comparing (RBufEnd - RBufStr) with rbufstat.
; 

ADD2TXRB:                         ; Adds characters to transmit ring buffer.
                                  ; The Tx ISR retrieves characters from the ring buffer.
    cli                           ; No interference from ISRs
    mov   ramptrl,  tx_headptrl     ; Load significant part of RAM pointer
    st    X+,       temp          ; Store value in first empty buffer location and increment pointer
	cpi   ramptrl,  LOW(TXRBUFEND)+1; Get the upper legal value for tx_endptrl + 1 and compare
	brne  A2TX_NOWRAP             ; Skip reset to start of buffer if not equal
	ldi   ramptrl,  LOW(TXRBUFSTR)   ;
A2TX_NOWRAP:                      ;
    mov   tx_headptrl, ramptrl      ; Store first empty buffer position
    inc   txrbstat                ; Increase character count in transmit ring buffer status
    sei                           ; Back to the real world
    ret

GET4RXRB:                         ; Retrieves characters from the receive ring buffer.
    cli                           ; This routine should only be called if rxrbstat > 0
	mov   ramptrl,  rx_tailptrl
	ld    temp,     X+             ; Retrieve value from first filled location and increment pointer
	cpi   ramptrl,  LOW(RXRBUFEND)+1  ; Compare
	brne  G4RX_NOWRAP              ; Skip next if pointer not outside buffer range
	ldi   ramptrl,  LOW(RXRBUFSTR) ; Reset to first buffer pointer
G4RX_NOWRAP:
	mov   rx_tailptrl, ramptrl     ; Store new pointer value
	dec   rxrbstat                 ; Decrease buffer contents size
	sei
	ret

GETSINVALUE:                       ; Get sinus value from table
;	                               ;  - Retrieval of next value from table
;	                               ;  - Check for end of table and wrap if needed
;								   ;  - In case of a wrap, toggle the wave polarity flag
;								   ;  - Check for halfway table and set ascending/descending
;								   ;  - flag which also depends on the polarity flag
;
    mov    ramptrl,  sintabptr
	ld     temp,  X+
	cpi    ramptrl,  LOW(SINUSTABLEEND)+1
	brne   GSV_NOWRAP
GSV_WRAP:
	ldi    ramptrl,  LOW(SINUSTABLE)
	rcall  BUSYLED					; Have the LED blink at cycle rate
	sbrs   flags,    tstWavPolfl    ; Skip next if Wave Polarity flag is set.
	rjmp   GSV_INVERT               ;  
GSV_NOINVERT:
	cbr    flags,    setWavPolfl
	rjmp   GSV_NOWRAP
GSV_INVERT:
	sbr    flags,    setWavPolfl
GSV_NOWRAP:
    mov    sintabptr,ramptrl
	cpi    ramptrl,  LOW(SINUSTABLE)+0x10 ; Compare current with prev. value
	brlo   GSV_1stHALF
GSV_2ndHALF:                        ; 
	sbrs   flags,    tstWavPolfl    ; Skip next if Wave Polarity flag is set (invert)
	sbr    flags,    setWavSlofl    ;  Set Wave Slope flag (descend)
	sbrc   flags,    tstWavPolfl    ; Skip next if Wave Polarity flag is clear (non-invert)
	cbr    flags,    setWavSlofl    ;  Clear Wave Slope flag (ascend)
	rjmp   GSV_CONT
GSV_1stHALF:                        ; 
	sbrs   flags,    tstWavPolfl    ; Skip next if WavePolarity flag is set (invert)
	cbr    flags,    setWavSlofl    ;  Clear Wave Slope flag (ascend)
	sbrc   flags,    tstWavPolfl    ; Skip next if Wave Polarity flag is clear (non-invert)
	sbr    flags,    setWavSlofl    ;  Set Wave Slope flag (descend)
GSV_CONT:                           ; Continue
	mov    tempWavVal, temp
	ret

MKMIDIPARAM:                        ; Adjust value to midi proportions
    lsr    temp                     ; Make it MIDI data compatibel
    lsr    temp                     ; Make it small enough for an inverted copy
	sbr    temp,  0x40              ; Set msb (MIDI data)
	sbrs   flags, tstWavPolfl       ; If the sinus table polarity flag is set
	com    temp                     ; invert the value.
	andi   temp,  0x7F              ; Make sure the upper bit is zero
	ret

CALCDUTCY:                          ; Calculates duty cycle part of the message rate timer
	mov   temp,      dutycycl
	lsl   temp
	lsl   temp
	sbrs  wavemode,  tstWModfl      ; Skip next if Wave Modulate flag is set
	rjmp  CD_CHKWP                  ;  tstWModfl clear
CD_CHKWS:                           ; Check Wave Slope flag
	sbrs  flags,     tstWavSlofl    ; Skip next if Wave Slope flag is set
	rjmp  CD_ISD                    ;  tstWavSlofl clear
CD_WSS:
	rjmp  CD_SUBD
CD_CHKWP:                           ; Check Wave Phase flag
	sbrs  flags,     tstWavPolfl    ; Skip next if Wave Polarity flag is set
	rjmp  CD_ISD                    ;  tstWavPolfl clear
CD_SUBD:
	cli
	ldi   tim0dutcy, 0x7F
	sub   tim0dutcy, temp
	sei
	rjmp  CD_EXIT
CD_ISD:
	mov   tim0dutcy, temp
	rjmp  CD_EXIT
CD_EXIT:
	ret

TRIACALC:
	subi  temp,   SINUSTABLE     ;  range of temp is now 0x00 - 0x1F
	sbrc  flags,  tstWavPolfl    ; Skip next if Wave Polarity flag is clear (positive)
	rjmp  TM_NEG
TM_POS:
	sbrc  flags,  tstWavSlofl    ; Skip next if Wave Slope flag is clear (ascending)
	rjmp  TM_POSDESC
TM_POSASC:                       ; 1. Positive half, ascending quarter. Add 0x10
	ldi   temp2,  0x10
	add   temp,   temp2
	rjmp  TM_EXIT
TM_POSDESC:                      ; 2. Positive half, descending quarter. Substract from 0x30
	ldi   temp2,  0x30
	sub   temp2,  temp
	mov   temp,   temp2
	rjmp  TM_EXIT
TM_NEG:                          
	sbrc  flags,  tstWavSlofl    ; Skip next if Wave Slope flag is clear(ascending)
	rjmp  TM_NEGDESC
TM_NEGASC:                       ; 4. Negative half, ascending quarter. Substract 0x10
	subi   temp,  0x10
	rjmp  TM_EXIT
TM_NEGDESC:                      ; 3. Negative half, descending quarter. Substract from 0x10
	ldi   temp2,  0x10
	sub   temp2,  temp
	mov   temp,   temp2
TM_EXIT:
	lsl   temp
	lsl   temp
	sbrc  temp,   7
	ldi   temp,   0x7F
	ret

WRITEEEPROM:
	sbic   EECR,  EEWE             ; Skip next if EEWE is clear
	rjmp   WRITEEEPROM             ; step 1
	cli
	out    EEAR,  temp2            ; step 2
	out    EEDR,  temp             ; step 3
	ldi    temp,  0b00000100
	out    EECR,  temp             ; step 4 EEMWE 1 and EEWE 0
	sbi    EECR,  EEWE             ; step 5
	sei
	ret

SAVEEEVALUE:
	ldi   temp,  0b11110000        ; Blink all LEDs durin write
	rcall HD3_WR
	cpi   editstat,    0
	breq  SEV_WAVTYP
	cpi   editstat,    1
	breq  SEV_WAVDS
	cpi   editstat,    2
	breq  SEV_MSGRATE
	cpi   editstat,    3
	breq  SEV_MCHAN
	cpi   editstat,    4
	breq  SEV_MCTRN
	rjmp  SEV_EXIT
SEV_WAVTYP:
	mov   temp,        wavemode
	ldi   temp2,       EEWAVEMODE
	rcall WRITEEEPROM
	rjmp  SEV_EXIT
SEV_WAVDS:
	mov   temp,        dutycycl
	ldi   temp2,       EEDUTYCYCL
	rcall WRITEEEPROM
	rjmp  SEV_EXIT
SEV_MSGRATE:
	mov   temp,        msgrate
	ldi   temp2,       EEMSGRATE
	rcall WRITEEEPROM
	rjmp  SEV_EXIT
SEV_MCHAN:
	mov   temp,        midichan
	ldi   temp2,       EEMIDICHAN
	rcall WRITEEEPROM
	rjmp  SEV_EXIT
SEV_MCTRN:
	mov   temp,        midictrn
	ldi   temp2,       EEMIDICRTN
	rcall WRITEEEPROM
SEV_EXIT:
	rcall DELAY5MS
	ldi   temp,  0b11111111        ; All LEDs out after write
	rcall HD3_WR
	ret

SENDMESSAGE:
	mov   temp,   midistat
	or    temp,   midichan
	rcall ADD2TXRB                 ; Status byte 
	mov   temp,   midictrn
	rcall ADD2TXRB                 ; Control Change number
	rcall GETVALUE
	rcall ADD2TXRB 
	sbi   UCSRB,  UDRIE            ; Enable it if needed.
	ret  

GETVALUE:
	sbrs  wavemode, tstQuadRatefl  ; Skip next if quadruple rate flag is set
	rjmp  GV_STDMSGRATE
	rcall GETSINVALUE
	rcall GETSINVALUE
	rcall GETSINVALUE
GV_STDMSGRATE:
	rcall GETSINVALUE              ; Placed here as some flags are used by non-sinus modes
	mov   temp2,  temp
	rcall CALCDUTCY                ; Uses flags set by GETSINVALUE

	mov   temp,   wavemode
	andi  temp,   0x03             ; Remove all upper bits, not to confuse jumptable
	cpi   temp,   sinusa
	breq  GV_SINUSMODE
	cpi   temp,   sinusb
	breq  GV_SINUSMODE
	cpi   temp,   square
	breq  GV_SQUAREMODE
	cpi   temp,   triang
	breq  GV_TRIANGMODE
	cpi   temp,   0x00           ; Safety option. Guarantees a valid midi message is produced
	ret              ; Should not happen. 
GV_SINUSMODE:
	mov   temp,   temp2
	rcall MKMIDIPARAM              ; Convert to MIDI size 00-7F and invert second half
	ret
GV_SQUAREMODE:
	ldi   temp,   0x7F
    sbrs  flags,  tstWavPolfl      ; Skip next if Wave Polarity flag is set
	ldi   temp,   0x00
	ret
GV_TRIANGMODE:
	mov   temp,   sintabptr        ; Mis-use the pointer value to create a triangle wave.
	rcall TRIACALC
	ret