/*
 * linux/kernel/chr_drv/sound/soundcard.c
 *
 * Soundcard driver for Linux
 */
/*
 * Copyright by Hannu Savolainen 1993-1996
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met: 1. Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer. 2.
 * Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <linux/config.h>


#include "sound_config.h"

#include <linux/major.h>

static int      chrdev_registered = 0;
static int      sound_major = SOUND_MAJOR;

static int      is_unloading = 0;

/*
 * Table for permanently allocated memory (used when unloading the module)
 */
caddr_t         sound_mem_blocks[1024];
int             sound_num_blocks = 0;

static int      soundcard_configured = 0;

static struct fileinfo files[SND_NDEVS];

#define DMA_MAP_UNAVAIL		0
#define DMA_MAP_FREE		1
#define DMA_MAP_BUSY		2


/*
 * /dev/sndstatus -device
 */
static char    *status_buf = NULL;
static int      status_len, status_ptr;
static int      status_busy = 0;

static int
put_status (char *s)
{
  int             l = strnlen (s, 256);

  if (status_len + l >= 4000)
    return 0;

  memcpy (&status_buf[status_len], s, l);
  status_len += l;

  return 1;
}

static int
put_status_int (unsigned int val, int radix)
{
  int             l, v;

  static char     hx[] = "0123456789abcdef";
  char            buf[11];

  if (!val)
    return put_status ("0");

  l = 0;
  buf[10] = 0;

  while (val)
    {
      v = val % radix;
      val = val / radix;

      buf[9 - l] = hx[v];
      l++;
    }

  if (status_len + l >= 4000)
    return 0;

  memcpy (&status_buf[status_len], &buf[10 - l], l);
  status_len += l;

  return 1;
}


static void
init_status (void)
{
  /*
   * Write the status information to the status_buf and update status_len.
   * There is a limit of 4000 bytes for the data.
   */

  status_ptr = 0;

  put_status ("Sound Driver Version 1.0 for Linux by Uwe Kirst, 15.9.96 \n");

  put_status ("Kernel: ");
  put_status (system_utsname.sysname);
  put_status (" ");
  put_status (system_utsname.nodename);
  put_status (" ");
  put_status (system_utsname.release);
  put_status (" ");
  put_status (system_utsname.version);
  put_status (" ");
  put_status (system_utsname.machine);
  put_status ("\n");

  if (!put_status ("\nInstalled drivers: \n"))
    return;
  
  {
    if (!put_status ("Type "))
      return;
    if (!put_status ( "AES/EBU input/output" ))
      return;
    if (!put_status (": "))
      return;
    if (!put_status ( "Digital Audio Interface" ))
      return;
    if (!put_status ("\n"))
      return;
  }

  if (!put_status ("\nCard config: \n"))
    return;

  {
    if (!put_status ( "Digital Audio Interface" ))
      return;
    if (!put_status (" at 0x"))
      return;
    if (!put_status_int (IO_BASE, 16))
      return;
    if (!put_status (" irq "))
      return;
    if (!put_status_int (IRQ_LEVEL, 10))
      return;
    if (!put_status ("\n"))
      return;
  }
 
}

static int
read_status (char *buf, int count)
{
  /*
   * Return at most 'count' bytes from the status_buf.
   */
  int             l, c;

  l = count;
  c = status_len - status_ptr;

  if (l > c)
    l = c;
  if (l <= 0)
    return 0;

  memcpy_tofs (&((buf)[0]), &status_buf[status_ptr], l);
  status_ptr += l;

  return l;
}


int
snd_ioctl_return (int *addr, int value)
{
  if (value < 0)
    return value;

  put_fs_long (value, (long *) &((addr)[0]));
  return 0;
}

static int
sound_read (inode_handle * inode, file_handle * file, char *buf, int count)
{
  int             dev;

  dev = MINOR (inode_get_rdev (inode));

  files[dev].flags = file_get_flags (file);

  switch (dev & 0x0f)
    {
    case SND_DEV_STATUS:
      return read_status (buf, count);
      break;
      
    case SND_DEV_DSP:
    case SND_DEV_DSP16:
    case SND_DEV_AUDIO:
      return audio_read (dev, &files[dev], buf, count);
      break;
    }
  return -EPERM;
}

static int
sound_write (inode_handle * inode, file_handle * file, const char *buf, int count)
{
  int             dev;

  dev = MINOR (inode_get_rdev (inode));

  files[dev].flags = file_get_flags (file);

  switch (dev & 0x0f)
    {
    case SND_DEV_DSP:
    case SND_DEV_DSP16:
    case SND_DEV_AUDIO:
      return audio_write (dev, &files[dev], buf, count);
      break;
    }
    return -EPERM;
}

static int
sound_lseek (inode_handle * inode, file_handle * file, off_t offset, int orig)
{
  return -EPERM;
}

static int
sound_open (inode_handle * inode, file_handle * file)
{
  int             dev, retval;
  struct fileinfo tmp_file;

  if (is_unloading)
    {
      printk ("Sound: Driver partially removed. Can't open device\n");
      return -EBUSY;
    }

  dev = MINOR (inode_get_rdev (inode));

  if (!soundcard_configured && dev != SND_DEV_CTL && dev != SND_DEV_STATUS)
    {
      printk ("SoundCard Error: The soundcard system has not been configured\n");
      return -ENXIO;
    }

  tmp_file.mode = 0;
  tmp_file.flags = file_get_flags (file);

  if ((tmp_file.flags & O_ACCMODE) == O_RDWR)
    tmp_file.mode = OPEN_READWRITE;
  if ((tmp_file.flags & O_ACCMODE) == O_RDONLY)
    tmp_file.mode = OPEN_READ;
  if ((tmp_file.flags & O_ACCMODE) == O_WRONLY)
    tmp_file.mode = OPEN_WRITE;

  switch (dev & 0x0f)
    {
    case SND_DEV_STATUS:
      if (status_busy)
        return -EBUSY;
      status_busy = 1;
      if ((status_buf = (char *) kmalloc (4000, GFP_KERNEL)) == NULL)
        return -EIO;
      status_len = status_ptr = 0;
      init_status ();
      retval = 0;
      break;
    
    case SND_DEV_DSP:
    case SND_DEV_DSP16:
    case SND_DEV_AUDIO:
     if ((retval = audio_open (dev, &tmp_file)) < 0)
       return retval;
     break;

    default: 
      printk ("Invalid minor device %d\n", dev);
      return -ENXIO;
    }
  


#ifdef MODULE
  MOD_INC_USE_COUNT;
#endif

  memcpy ((char *) &files[dev], (char *) &tmp_file, sizeof (tmp_file));
  return retval;
}

static void
sound_release (inode_handle * inode, file_handle * file)
{
  int             dev;

  dev = MINOR (inode_get_rdev (inode));

  files[dev].flags = file_get_flags (file);

  switch (dev & 0x0f)
    {
    case SND_DEV_STATUS:
      if (status_buf)
        kfree (status_buf);
      status_buf = NULL;
      status_busy = 0;
      break;
       
    case SND_DEV_DSP:
    case SND_DEV_DSP16:
    case SND_DEV_AUDIO:
      audio_release (dev, &files[dev]);
      break;
  
    default: 
      printk ("Sound error: Releasing unknown device 0x%02x\n", dev);
    }
#ifdef MODULE
  MOD_DEC_USE_COUNT;
#endif
}

static int
sound_ioctl (inode_handle * inode, file_handle * file,
	     unsigned int cmd, unsigned long arg)
{
  int             dev, err;

  dev = MINOR (inode_get_rdev (inode));

  files[dev].flags = file_get_flags (file);

  if (_IOC_DIR (cmd) != _IOC_NONE)
    {
      /*
         * Have to validate the address given by the process.
       */
      int             len;

      len = _IOC_SIZE (cmd);

      if (_IOC_DIR (cmd) & _IOC_WRITE)
	{
	  if ((err = verify_area (VERIFY_READ, (void *) arg, len)) < 0)
	    return err;
	}

      if (_IOC_DIR (cmd) & _IOC_READ)
	{
	  if ((err = verify_area (VERIFY_WRITE, (void *) arg, len)) < 0)
	    return err;
	}

    }

  err = audio_ioctl (dev, &files[dev], cmd, (caddr_t) arg, 0);

  return err;
}

static int
sound_select (inode_handle * inode, file_handle * file, int sel_type, select_table_handle * wait)
{
  int             dev;

  dev = MINOR (inode_get_rdev (inode));

  files[dev].flags = file_get_flags (file);

  DEB (printk ("sound_select(dev=%d, type=0x%x)\n", dev, sel_type));

  switch (dev & 0x0f)
    {

    case SND_DEV_DSP:
    case SND_DEV_DSP16:
    case SND_DEV_AUDIO:
      return audio_select (dev, &files[dev], sel_type, wait);
      break;

    default:
      return 0;
    }

  return 0;
}

static int
sound_mmap (inode_handle * inode, file_handle * file, vm_area_handle * vma)
{
  return -EINVAL;
}

static struct file_operation_handle sound_fops =
{
  sound_lseek,
  sound_read,
  sound_write,
  NULL,				/* sound_readdir */
  sound_select,
  sound_ioctl,
  sound_mmap,
  sound_open,
  sound_release
};

void
soundcard_init (void)
{
#ifndef MODULE
  module_register_chrdev (sound_major, "sound", &sound_fops);
  chrdev_registered = 1;
#endif

  soundcard_configured = 1;
  audio_init();   
}

static unsigned int irqs = 0;

#ifdef MODULE
static void
free_all_irqs (void)
{
  int             i;

  for (i = 0; i < 31; i++)
    if (irqs & (1ul << i))
      {
	printk ("Sound warning: IRQ%d was left allocated - fixed.\n", i);
	snd_release_irq (i);
      }
  irqs = 0;
}

char            kernel_version[] = UTS_RELEASE;

#endif

int
init_module (void)
{
  int             err;

  if (connect_wrapper (WRAPPER_VERSION) < 0)
    {
      printk ("Sound: Incompatible kernel (wrapper) version\n");
      return -EINVAL;
    }

  err = module_register_chrdev (sound_major, "sound", &sound_fops);
  if (err)
    {
      printk ("sound: driver already loaded/included in kernel\n");
      return err;
    }

  chrdev_registered = 1;
  soundcard_init ();

  if (sound_num_blocks >= 1024)
    printk ("Sound warning: Deallocation table was too small.\n");

  return 0;
}

#ifdef MODULE


void
cleanup_module (void)
{
  int             i;

  if (MOD_IN_USE)
    {
      return;
    }

  if (chrdev_registered)
    module_unregister_chrdev (sound_major, "sound");

  audio_unload ();

  for (i = 0; i < sound_num_blocks; i++)
    kfree (sound_mem_blocks[i]);

  free_all_irqs ();		/* If something was left allocated by accident */

}
#endif

void
tenmicrosec (int *osp)
{
  int             i;

  for (i = 0; i < 16; i++)
    inb (0x80);
}

int
snd_set_irq_handler (int interrupt_level, void (*hndlr) (int, void *, struct pt_regs *), char *name, int *osp)
{
  int             retcode;

  retcode = request_irq (interrupt_level, hndlr, 0 /* SA_INTERRUPT */ , name, NULL);
  if (retcode < 0)
    {
      printk ("Sound: IRQ%d already in use\n", interrupt_level);
    }
  else
    irqs |= (1ul << interrupt_level);

  return retcode;
}

void
snd_release_irq (int vect)
{
  if (!(irqs & (1ul << vect)))
    return;

  irqs &= ~(1ul << vect);
  free_irq (vect, NULL);
}


#ifdef KMALLOC_DMA_BROKEN
fatal_error__This_version_is_not_compatible_with_this_kernel;
#endif

void
conf_printf (char *name, int base, int irq, int dma, int dma2)
{
  printk ("<%s> at 0x%03x", name, base);

  if (irq)
    printk (" irq %d", (irq > 0) ? irq : -irq);

  if (dma != -1 || dma2 != -1)
    {
      printk (" dma %d", dma);
      if (dma2 != -1)
	printk (",%d", dma2);
    }

  printk ("\n");
}
