/* @file TwoAxisControl.ino
|| @version 1.0
|| @author Fred Jan Kraan
|| @contact fjkraan@xs4all.nl
||
|| @description
|| | Controls two servos with a key pad, including border calibration
|| | and four presets. Keys 2, 4, 6, 8 are up, left, right and down respectively.
|| | A, B, C, D are preset recalls. *A, *B, *C and *D stores current position.
|| #
*/
#define VERSION 0.1

#include <Keypad.h>
#include <Servo.h> 
#include <EEPROM.h>

const byte ROWS = 4; //four rows
const byte COLS = 4; //three columns
char keys[ROWS][COLS] = {
  {'1','2','3','A'},
  {'4','5','6','B'},
  {'7','8','9','C'},
  {'*','0','#','D'}
};
byte rowPins[ROWS] = {5, 4, 3, 2}; //connect to the row pinouts of the kpd
byte colPins[COLS] = {9, 8, 7, 6}; //connect to the column pinouts of the kpd

Keypad kpd = Keypad( makeKeymap(keys), rowPins, colPins, ROWS, COLS );


Servo upDown;  // create servo object to control a servo 
Servo leftRight;

#define MAGIC         42
#define UPDOWN_MIN    1
#define UPDOWN_MAX    180
#define LEFTRIGHT_MIN 1
#define LEFTRIGHT_MAX 180
#define EE_MAGIC     0
#define EE_DOWN_MIN  1
#define EE_UP_MAX    2
#define EE_LEFT_MIN  3
#define EE_RIGHT_MAX 4
#define EE_STEPRATE  5

#define EE_A_COOR_UD 6
#define EE_A_COOR_LR 7
#define EE_B_COOR_UD 8
#define EE_B_COOR_LR 9
#define EE_C_COOR_UD 10
#define EE_C_COOR_LR 11
#define EE_D_COOR_UD 12
#define EE_D_COOR_LR 13

uint8_t upDownPos = UPDOWN_MAX / 2;
uint8_t leftRightPos = LEFTRIGHT_MAX / 2;
uint8_t upMaxPos;
uint8_t downMinPos;
uint8_t leftMinPos;
uint8_t rightMaxPos;

#define SERIALBUFSIZE         11
char serialBuffer[SERIALBUFSIZE];
byte setBufPointer = 0;
char* unsupported = "unsupported";

unsigned long loopCount;
unsigned long startTime;
String msg;


void setup() {
    Serial.begin(9600);
    loopCount = 0;
    startTime = millis();
    msg = "";
    upDown.attach(10);
    leftRight.attach(11);
    
    servoCalibration();
    Serial.println("MultiKeyMod 0.1");
}


void loop() {
    loopCount++;
    if ( (millis()-startTime)>5000 ) {
 //       Serial.print("Average loops per second = ");
 //       Serial.println(loopCount/5);
        startTime = millis();
        loopCount = 0;
    }
    delay(20);

    // Fills kpd.key[ ] array with up-to 10 active keys.
    // Returns true if there are ANY active keys.
    if (kpd.getKeys())
    {
        for (int i=0; i<LIST_MAX; i++)   // Scan the whole key list.
        {
            if ( kpd.key[i].stateChanged )   // Only find keys that have changed state.
            {
                switch (kpd.key[i].kstate) {  // Report active key state : IDLE, PRESSED, HOLD, or RELEASED
                    case PRESSED:
                    msg = " PRESSED.";
                break;
                    case HOLD:
                    msg = " HOLD.";
                break;
                    case RELEASED:
                    msg = " RELEASED.";
                break;
                    case IDLE:
                    msg = " IDLE.";
                }
            }
        }
    }

    if (kpd.isPressed('A') && !isStillPressed('*')) {
      Serial.print("A");
      upDownPos = EEPROM.read(EE_A_COOR_UD);
      leftRightPos = EEPROM.read(EE_A_COOR_LR);
      upDown.write(upDownPos);
      leftRight.write(leftRightPos);
    }
    if (kpd.isPressed('B') && !isStillPressed('*')) {
      Serial.print("B");
      upDownPos = EEPROM.read(EE_B_COOR_UD);
      leftRightPos = EEPROM.read(EE_B_COOR_LR);
      upDown.write(upDownPos);
      leftRight.write(leftRightPos);
    }
    if (kpd.isPressed('C') && !isStillPressed('*')) {
      Serial.print("C");
      upDownPos = EEPROM.read(EE_C_COOR_UD);
      leftRightPos = EEPROM.read(EE_C_COOR_LR);
      upDown.write(upDownPos);
      leftRight.write(leftRightPos);
    }
    if (kpd.isPressed('D') && !isStillPressed('*')) {
      Serial.print("D");
      upDownPos = EEPROM.read(EE_D_COOR_UD);
      leftRightPos = EEPROM.read(EE_D_COOR_LR);
      upDown.write(upDownPos);
      leftRight.write(leftRightPos);
    }
    if (isStillPressed('2')) {
      updateUp();
    }
    if (isStillPressed('4')) {
      updateLeft();
    }
    if (isStillPressed('6')) {
      updateRight();
    }
    if (isStillPressed('8')) {
      updateDown();
    }
    if (isStillPressed('*')) {
       if (isStillPressed('A')) {
         EEPROM.write(EE_A_COOR_UD, upDownPos);
         EEPROM.write(EE_A_COOR_LR, leftRightPos);
         Serial.println("Preset A saved");
       }
       if (isStillPressed('B')) {
         EEPROM.write(EE_B_COOR_UD, upDownPos);
         EEPROM.write(EE_B_COOR_LR, leftRightPos);
         Serial.println("Preset B saved");
       }
       if (isStillPressed('C')) {
         EEPROM.write(EE_C_COOR_UD, upDownPos);
         EEPROM.write(EE_C_COOR_LR, leftRightPos);
         Serial.println("Preset C saved");
       }
       if (isStillPressed('D')) {
         EEPROM.write(EE_D_COOR_UD, upDownPos);
         EEPROM.write(EE_D_COOR_LR, leftRightPos);
         Serial.println("Preset D saved");
       }
    }
    commandCollector();
}  // End loop


bool isStillPressed(char keyChar) {
    for (byte i=0; i<LIST_MAX; i++) {
	if ( kpd.key[i].kchar == keyChar ) {
	    if ( (kpd.key[i].kstate == PRESSED || kpd.key[i].kstate == HOLD) )
		return true;
            }
    }
    return false;	// Not pressed.
}

void updateUp() {
//  Serial.print("U");
  upDownPos++;
  if (upDownPos > upMaxPos) { upDownPos = upMaxPos; }
  upDown.write(upDownPos);
}

void updateDown() {
//  Serial.print("D");
  upDownPos--;
  if (upDownPos < downMinPos) { upDownPos = downMinPos; }
  upDown.write(upDownPos);
}

void updateLeft() {
//  Serial.print("L");
  leftRightPos--;
  if (leftRightPos < leftMinPos) { leftRightPos = leftMinPos; }
  leftRight.write(leftRightPos);
}

void updateRight() {
//  Serial.print("R");
  leftRightPos++;
  if (leftRightPos > rightMaxPos) { leftRightPos = rightMaxPos; }
  leftRight.write(leftRightPos);
}

void commandCollector() {
  if (Serial.available() > 0) {
    int inByte = Serial.read();
    switch(inByte) {
    case '.':
    case '\r':
    case '\n':
      commandInterpreter();
      clearSerialBuffer();
      setBufPointer = 0;
      break;
    default:
      serialBuffer[setBufPointer] = inByte;
      setBufPointer++;
      if (setBufPointer >= SERIALBUFSIZE) {
        Serial.println("Serial buffer overflow. Cleanup.");
        clearSerialBuffer();
        setBufPointer = 0;
      }
    }
  }
}

void commandInterpreter() {
  byte bufByte = serialBuffer[0];
  switch(bufByte) {
    case 'D':  // get/set date
    case 'd':
      dArgInterpreter();
      break; 
    case 'P':  // get/set date
    case 'p':
      pArgInterpreter();
      break; 
    case 'L':  // get/set date
    case 'l':
      lArgInterpreter();
      break; 
    case 'R':  // get/set date
    case 'r':
      rArgInterpreter();
      break; 
    case 'U':  // get/set evening switch time
    case 'u':
      uArgInterpreter();
      break;
    case 'X':
    case 'x':
      xArgInterpreter();
      break;
    case 'H':  // help
    case 'h':
    case '?':  // help
      usage();
      break; 
    default:
      Serial.print(bufByte);
      Serial.print(" ");
      Serial.println(unsupported);
      return;
  }
}

void dArgInterpreter() {
  if (setBufPointer == 1) {
    Serial.print(downMinPos);
    Serial.println();
  } else if (setBufPointer == 4) {
    downMinPos  = (serialBuffer[1]-'0') * 100 + (serialBuffer[2]-'0') * 10 +
                  (serialBuffer[3]-'0');
    EEPROM.write(EE_DOWN_MIN, downMinPos); 
  } else {
    Serial.println(unsupported);
  }
} 

void uArgInterpreter() {
  if (setBufPointer == 1) {
    Serial.print(upMaxPos);
    Serial.println();
  } else if (setBufPointer == 4) {
    upMaxPos  = (serialBuffer[1]-'0') * 100 + (serialBuffer[2]-'0') * 10 +
                  (serialBuffer[3]-'0');
    EEPROM.write(EE_UP_MAX, upMaxPos); 
  } else {
    Serial.println(unsupported);
  }
} 

void vArgInterpreter() {
  if (setBufPointer == 1) {
    Serial.print(VERSION);
    Serial.println();  
  } else {
    Serial.println(unsupported);
  } 
}

void lArgInterpreter() {
  if (setBufPointer == 1) {
    Serial.print(leftMinPos);
    Serial.println();
  } else if (setBufPointer == 4) {
    leftMinPos  = (serialBuffer[1]-'0') * 100 + (serialBuffer[2]-'0') * 10 +
                  (serialBuffer[3]-'0');
    EEPROM.write(EE_LEFT_MIN, leftMinPos); 
  } else {
    Serial.println(unsupported);
  }
} 

void pArgInterpreter() {
  if (setBufPointer == 1) {
      Serial.print("U/D - L/R: ");
      Serial.print(upDownPos, DEC);
      Serial.print(" - ");
      Serial.print(leftRightPos, DEC);
      Serial.println();
  } else {
    Serial.println(unsupported);
  }
}

void rArgInterpreter() {
  if (setBufPointer == 1) {
    Serial.print(rightMaxPos);
    Serial.println();
  } else if (setBufPointer == 4) {
    rightMaxPos  = (serialBuffer[1]-'0') * 100 + (serialBuffer[2]-'0') * 10 +
                  (serialBuffer[3]-'0');
    EEPROM.write(EE_RIGHT_MAX, rightMaxPos); 
  } else {
    Serial.println(unsupported);
  }
} 

void sArgInterpreter() {
  if (setBufPointer == 1) {
 //   Serial.print(stepRate);
    Serial.println();
    
  } else {
    Serial.println(unsupported);
  }
}  

void xArgInterpreter() {
  if (setBufPointer == 1) {
    for (uint8_t i = 0; i < 15; i++) {
      EEPROM.write(i, 255);
      Serial.print(".");
    }
    Serial.println("Done erasing EEPROM");
  } else {
    Serial.println(unsupported);
  }
}   

void usage() {
  Serial.println("D[ddd] - Minimal down value");
  Serial.println("H      - This help text");
  Serial.println("L[lll] - Minimal left value");
  Serial.println("R[rrr] - Maximal right value");
  Serial.println("U[uuu] - Maximal up value");
  Serial.println("V      - Version number");
  Serial.println("X      - Reset EEPROM");
  Serial.println("?      - This help text"); 
}

void clearSerialBuffer() {
  byte i;
  for (i = 0; i < SERIALBUFSIZE; i++) {
    serialBuffer[i] = 0;
  }
}

void servoCalibration() {
    if (EEPROM.read(EE_MAGIC) == MAGIC) { 
      upMaxPos    = EEPROM.read(EE_UP_MAX);
      downMinPos  = EEPROM.read(EE_DOWN_MIN);
      leftMinPos  = EEPROM.read(EE_LEFT_MIN);
      rightMaxPos = EEPROM.read(EE_RIGHT_MAX);
      Serial.println("Values read from EEPROM");
    } else { // Only used at first startup
      EEPROM.write(EE_UP_MAX,    UPDOWN_MAX);
      EEPROM.write(EE_DOWN_MIN,  UPDOWN_MIN);
      EEPROM.write(EE_LEFT_MIN,  LEFTRIGHT_MIN);
      EEPROM.write(EE_RIGHT_MAX, LEFTRIGHT_MAX);
      EEPROM.write(EE_MAGIC, MAGIC);
      EEPROM.write(EE_A_COOR_UD, 1);
      EEPROM.write(EE_A_COOR_LR, 1);
      EEPROM.write(EE_B_COOR_UD, 1);
      EEPROM.write(EE_B_COOR_LR, 1);
      EEPROM.write(EE_C_COOR_UD, 1);
      EEPROM.write(EE_C_COOR_LR, 1);
      EEPROM.write(EE_D_COOR_UD, 1);
      EEPROM.write(EE_D_COOR_LR, 1);
      
      Serial.println("Default values written to EEPROM");
    }
}
