/*
 * MC6803 pin exersizer
 */

#define NAME   "MC6803exer V"
#define VERSION 0.2

#include "6803pins.h" 

#define SERIALBUFSIZE         50
char serialBuffer[SERIALBUFSIZE];
byte setBufPointer = 0;

#define LED 13

uint16_t repeatRate = 1 < 9;
#define RECORDSIZE 16
#define DATARECORDTYPE 0

#define DUMPPAGE 0x00FF
unsigned int lastEndAddress = 0;

unsigned int addressOffset = 0;
#define E_PULSE  10 // in microseconds
#define AS_PULSE 1 // in microseconds

#define PORT_OUT 0xFF;
#define PORT_IN  0x00;

// core routines

void setup() {
  Serial.begin(9600);
  Serial.print(NAME);
  Serial.println(VERSION, 1);
  
  pinMode(LED, OUTPUT);
  
  pinMode(MCRES,   INPUT_PULLUP);
  pinMode(MCIRQ,   INPUT_PULLUP);
  pinMode(MCNMI,   INPUT_PULLUP);

  digitalWrite(MCE,  LOW);
  digitalWrite(MCAS, LOW);
  digitalWrite(MCRW, HIGH);
  pinMode(MCE,     OUTPUT);
  pinMode(MCAS,    OUTPUT);
  pinMode(MCRW,    OUTPUT);
 
  DDRA  = PORT_IN;  // address LSB /data input
  DDRC  = PORT_OUT; // address MSB output
  PORTK = PORT_OUT; // control out bits high
  DDRK  = PORT_OUT; // control out bits output
  DDRL  = PORT_IN;  // P1x bus input
  PORTL = PORT_OUT; // P1x bus pull ups
  
//  onlineReadMode();
  delay(1000);  
}  

void loop() {
  byte refreshAddress = 0;
  commandCollector();
}  


void clearSerialBuffer() {
  byte i;
  for (i = 0; i < SERIALBUFSIZE; i++) {
    serialBuffer[i] = 0;
  }
}

void commandCollector() {
  if (Serial.available() > 0) {
    int inByte = Serial.read();
    switch(inByte) {
    case '.':
//    case '\r':
    case '\n':
      commandInterpreter();
      clearSerialBuffer();
      setBufPointer = 0;
      break;
    case '\r':
      break;  // ignore carriage return
    default:
      serialBuffer[setBufPointer] = inByte;
      setBufPointer++;
      if (setBufPointer >= SERIALBUFSIZE) {
        Serial.println("Serial buffer overflow. Cleanup.");
        clearSerialBuffer();
        setBufPointer = 0;
      }
    }
  }
}

void commandInterpreter() {
    byte bufByte = serialBuffer[0];
    
    switch(bufByte) {
      case 'A':
      case 'a':
        setAddress();
        break;
      case 'B':
      case 'b':
        blinkPin();
        break;
      case 'D':  // dump memory
      case 'd':
        dumpMemory();
        break;
      case 'H':  // help
      case 'h':
      case '?':  // help
  //      Serial.println("F?:");
        usage();
        break; 
      case 'I':
      case 'i':
        generateDataRecords();
        generateEndRecord();
        break;
      case 'M':  // memory address read/write
      case 'm':
        readWriteMemory();
        break; 
      case 'P':
      case 'p':
        viewPorts();
        break;
      case 'Q':  // repeatRate
      case 'q':
        setRepeatRate();
        break; 
  //    case 'S':
  //    case 's':
  //      setValue(); // fill memory range with a value
  //      break;
      case 'W':
      case 'w':
        writePin();
        break;
      default:
        Serial.print(bufByte);
        Serial.print(" ");
        Serial.println("unsupported");
        return;
    }
}

void usage() {
    Serial.print("-- ");
    Serial.print(NAME);
    Serial.print(VERSION, 1);
    Serial.println(" command set --");
    Serial.println("Aaaaa          - Set address bus to value aaaa");
    Serial.println("Bpp or B#ss    - Blink pin p (in hex) or symbol: AD0-AD7,A8-AF,RW,E,AS, P10-P17, P20-P24");
    Serial.println("D[ssss[-eeee]|+] - Dump memory from ssss to eeee (default 256 bytes)");
    Serial.println("H              - This help text");
    Serial.println("Issss-eeee     - Generate hex intel data records");
    Serial.println("MRaaaa[+]      - Read memory address aaaa, optionally repeating");
    Serial.println("MWaaaa vv[+]   - Write vv to address aaaa, optionally repeating");
    Serial.println("Pp[+]          - View port/pins; A, C, L, c(omm), B/D/G");
    Serial.println("Qn             - Blink/Repeat rate; 1, 2, 4, ..., 32678 ms (n=0-9,A-F)");
    Serial.println("S[+]           - Signal view; RES, INT, NMI");
    Serial.println("V              - View data bus, pins INT, NMI, WAIT, BUSRQ, RESET");
    Serial.println("Wpp v or W#ss v - Write pin (in hex) or symbol: AD0-AD7,A8-AF,RW,E,AS; values 0, 1");
    Serial.println("?              - This help text"); 
}

void dumpMemory() {
    unsigned int startAddress;
    unsigned int endAddress;
    bool repeatMode = 0;
    if (setBufPointer == 1 ) { // Automatic mode, dumping next page
        startAddress   = lastEndAddress == 0 ? 0 : lastEndAddress + 1;
        endAddress     = startAddress + DUMPPAGE;
        lastEndAddress = endAddress;
    } else if (setBufPointer == 2 && serialBuffer[1] == '+') { // continiously reading one page
        startAddress   = lastEndAddress - DUMPPAGE;
        endAddress     = startAddress + DUMPPAGE;
        lastEndAddress = endAddress;
        repeatMode = 1;
    } else if (setBufPointer == 5) { // dumping specified page
        startAddress   = get16BitValue(1);
        endAddress     = startAddress + DUMPPAGE;
        lastEndAddress = endAddress;
    } else if (setBufPointer == 10) { // dumping specified memory range
        startAddress   = get16BitValue(1);
        endAddress     = get16BitValue(6);
        lastEndAddress = endAddress;
    } else {
        Serial.println("unsupported"); 
    }
    unsigned char asChars[17];
    unsigned char *asCharsP = &asChars[0];
    unsigned char positionOnLine;
    asChars[16] = 0;
    do {
  //    printWord(startAddress);
  //    Serial.print("-");
  //    printWord(endAddress);
  //    Serial.println();
        unsigned int i, data;
  
        for (i = startAddress; i <= endAddress; i++) {
            positionOnLine = i & 0x0F;
            if (positionOnLine == 0) {
                printWord(i);   // Address at start of line
                Serial.print(": ");
            }
            data = readByte(i);
            printByte(data);   // actual value in hex
            // fill an array with the ASCII part of the line
            asChars[positionOnLine] = (data >= ' ' && data <= '~') ? data : '.';
            if ((i & 0x03) == 0x03) Serial.print(" ");
            if ((i & 0x0F) == 0x0F) {
                Serial.print (" ");
                printString(asCharsP); // print the ASCII part
                Serial.println("");
            }
        }
        Serial.println();
        delay(repeatRate);
        if (stopIt()) repeatMode = 0;
    } while (repeatMode);
}

unsigned int readByte(unsigned int address) { // for muxed bus
    unsigned int data = 0;
    unsigned int addressLSB = address & 0xFF;
    unsigned int addressMSB = address >> 8;
//An output
    DDRC = PORT_OUT;
//ADn output (address LSB)
    DDRA = PORT_OUT;
//E low
    digitalWrite(MCE, LOW);
//An write
    PORTC = addressMSB;
//ADn write
    PORTA = addressLSB;
//R/W high
    digitalWrite(MCRW, HIGH);
//AS high
    digitalWrite(MCAS, HIGH);
//delay AS_PULSE
    delayMicroseconds(AS_PULSE);
//AS low
    digitalWrite(MCAS, LOW);
//ADn input
    DDRA = PORT_IN;
//E high
    digitalWrite(MCE, HIGH);
//ADn read (data)
    data = PINA;
//E low
    digitalWrite(MCE, LOW);
    
    return data; 
}

void writeByte(unsigned int address, unsigned int value) { // muxed bus
    unsigned int addressLSB = address & 0xFF;
    unsigned int addressMSB = address >> 8;
//An output
    DDRC = PORT_OUT;
//ADn output
    DDRA = PORT_OUT;
//An write
    PORTC = addressMSB;
//ADn write (address)
    PORTA = addressLSB;
//E low
    digitalWrite(MCE, LOW);
//R/W low
    digitalWrite(MCRW, LOW);
//AS high
    digitalWrite(MCAS, HIGH);
//delay AS_PULSE
    delayMicroseconds(AS_PULSE);
//AS low
    digitalWrite(MCAS, LOW);
//ADn write (data)
    PORTA = value;
//E high
     digitalWrite(MCE, HIGH);
//delay E_PULSE
     delayMicroseconds(E_PULSE);
//E low
    digitalWrite(MCE, LOW);  
//ADn input
    DDRA = PORT_IN;
}


void printByte(unsigned char data) {
    unsigned char dataMSN = data >> 4;
    unsigned char dataLSN = data & 0x0F;
    Serial.print(dataMSN, HEX);
    Serial.print(dataLSN, HEX);
}

void printWord(unsigned int data) {
    printByte(data >> 8);
    printByte(data & 0xFF);
}

void dataBusReadMode() {
   DDRL  = 0x00;  // read mode
}

void dataBusWriteMode() {
   DDRL  = 0xFF;  // write mode
}

void blinkPin() {
    int pin;
    Serial.print("B");
    if (serialBuffer[1] == '#') {
        Serial.print("#");
        pin = getPinBySymbol();    
    } else {
        pin  = get8BitValue(1);
        Serial.print(serialBuffer[1]);
        Serial.print(serialBuffer[2]);
        Serial.print(" ");
    }
    Serial.print("Pin: ");
    Serial.println(pin, DEC);  
    pinMode(pin, OUTPUT);
    while(1) {
        digitalWrite(pin, !digitalRead(pin));
        delay(repeatRate);
        if (stopIt()) return;
    }
}

void writePin() {
    unsigned char pin;
    bool value;
    Serial.print("Setting pin ");
    if (serialBuffer[1] == '#') {
        pin = getPinBySymbol();    
        value = (serialBuffer[5] == 0x30);
        Serial.print(serialBuffer[2]);
        Serial.print(serialBuffer[3]);
        Serial.print(" (");
        Serial.print(pin, DEC);
        Serial.print(") to ");
        Serial.println(serialBuffer[5]);
    } else {
        pin  = get8BitValue(1);
        value = (serialBuffer[4] == 0x30);
        Serial.print(serialBuffer[1]);
        Serial.print(serialBuffer[2]);
        Serial.print(" (");
        Serial.print(pin, DEC);
        Serial.print(") to ");
        Serial.println(serialBuffer[4]);    
    }
    digitalWrite(pin, value ? 0 : 1);
}

unsigned char getPinBySymbol() {
    // B[0-F][0-F], B#A[8-F], B#AD[0-7], B#RW, B#E, B#AS, B#P1[0-7], B#P2[0-4]
    if (serialBuffer[2] == 'A' && serialBuffer[3] == 'D' && serialBuffer[4] <= '7') {
        Serial.print("AD");
        Serial.write(serialBuffer[4]);
        Serial.print(" ");
        return serialBuffer[4] - '0' + MCAD0;
    } else if (serialBuffer[2] == 'A' && serialBuffer[3] > '7' && serialBuffer[3] <= 'F') {
        int i = serialBuffer[3] - '8';
        i = MCA8 - i;
        Serial.print("A");
        Serial.write(serialBuffer[3]);
        Serial.print(" ");
        return (i <= 28) ? i + 7 : i;
    } else if (serialBuffer[2] == 'P' && serialBuffer[3] == '1') {
        int i = serialBuffer[4] - '0';
        Serial.print("P1");
        Serial.write(serialBuffer[4]);
        return i + MCP10;
    } else if (serialBuffer[2] == 'P' && serialBuffer[3] == '2') {
        int i = serialBuffer[4] - '0';
        Serial.print("P2");
        Serial.write(serialBuffer[4]);
        return i + MCP20;
    } else if (serialBuffer[2] == 'R' && serialBuffer[3] == 'W') {
        Serial.print("RW ");  
        return MCRW;
    } else if (serialBuffer[2] == 'A' && serialBuffer[3] == 'S') {
        Serial.print("AS ");  
        return MCAS;
    } else if (serialBuffer[2] == 'E') {
        Serial.print("E ");  
        return MCE;
    } else {
        Serial.println(" unknown symbol");
    }
    return 0;
}

void printBin(unsigned char value) {
    Serial.print((value & 0b10000000) ? "1" : "0");
    Serial.print((value & 0b01000000) ? "1" : "0");
    Serial.print((value & 0b00100000) ? "1" : "0");
    Serial.print((value & 0b00010000) ? "1" : "0");
    Serial.print((value & 0b00001000) ? "1" : "0");
    Serial.print((value & 0b00000100) ? "1" : "0");
    Serial.print((value & 0b00000010) ? "1" : "0");
    Serial.print((value & 0b00000001) ? "1" : "0");  
}

void printString(unsigned char *asCharP) {
    unsigned char i = 0;
    while(asCharP[i] != 0) {
        Serial.write(asCharP[i]); 
        i++;
    }
}

void readWriteMemory() {
    // MRaaaa[+], MWaaaa vv[+]
    uint16_t address;
    bool repeatMode = 0;
    address  = get16BitValue(2);
    if (serialBuffer[1] == 'R' || serialBuffer[1] == 'r') {
        if (setBufPointer == 7 && serialBuffer[6] == '+') {
            repeatMode = 1;
        }
        dataBusReadMode();
        do {
            Serial.print("MRD ");
            Serial.print(address, HEX);
            Serial.print(": ");
            Serial.println(readByte(address), HEX);
            if (stopIt()) {
                return;
            }
            delay(repeatRate);
            if (stopIt()) repeatMode = 0;
        } while (repeatMode);
    /*  } else if (serialBuffer[1] == 'W' || serialBuffer[1] == 'w') {
        uint8_t data;
        data  = get8BitValue(7);
        if (setBufPointer == 10 && serialBuffer[9] == '+') {
            repeatMode = 1;
        }
        dataBusWriteMode();
        do {
            Serial.print("MWR ");
            Serial.print(address, HEX);
            Serial.print(": ");
            Serial.println(data, HEX);
            writeByte(address, data);
            if (stopIt()) {
                return;
            }
            delay(repeatRate);  
            if (stopIt()) repeatMode = 0;
        } while (repeatMode); */
    } else {
      Serial.print("not supported");
      return; 
    }
}

bool stopIt() {
    if (Serial.available() > 0) {
        clearSerialBuffer();
        setBufPointer = 0;
        return true;
    }
    return false;
}

void setRepeatRate() {
    if (setBufPointer == 2) {
        byte value = serialBuffer[1] - '0';
        value = (value > 9) ? (value - 7) : value;
        repeatRate = 1 << value;
        Serial.print("Repeat rate set to ");
        Serial.print(repeatRate, DEC);
        Serial.println(" ms");
    } else {
        Serial.println("not supported");
    }
}

void setAddress() {
    // Aaaaa
    uint16_t address;
    bool repeatMode = 0;
    address  = get16BitValue(1);
  
    unsigned int addressLSB = address & 0xFF;
    unsigned int addressMSB = address >> 8;
    PORTA = addressLSB;
    PORTC = addressMSB;
    Serial.print("A:");
    Serial.println(address,HEX);
}

void generateDataRecords() {
    unsigned int startAddress;
    unsigned int endAddress;
    startAddress  = get16BitValue(1);
    endAddress  = get16BitValue(6);
    printWord(startAddress);
    Serial.print("-");
    printWord(endAddress);
    Serial.println();
  
    unsigned int i, j;
    unsigned char addressMSB, addressLSB, data;
    unsigned char sumCheckCount = 0;
  
    dataBusReadMode();  
    for (i = startAddress; i < endAddress; i += RECORDSIZE) {
        sumCheckCount = 0;
        Serial.print(":");
        printByte(RECORDSIZE);  
        sumCheckCount -= RECORDSIZE;
        addressMSB = i >> 8;
        addressLSB = i & 0xFF;
        printByte(addressMSB);
        printByte(addressLSB);
        sumCheckCount -= addressMSB;
        sumCheckCount -= addressLSB;
        printByte(DATARECORDTYPE);
        sumCheckCount -= DATARECORDTYPE;
        for (j = 0; j < RECORDSIZE; j++) {
            data = readByte(i + j);
            printByte(data);
            sumCheckCount -= data;
      }
      printByte(sumCheckCount);
      Serial.println();
    }
}

void generateEndRecord() {
  Serial.println(":00000001FF");
}

unsigned int get16BitValue(byte index) {
    byte i = index;
    unsigned address;
    address  = getNibble(serialBuffer[i++]) * (1 << 12);
    address += getNibble(serialBuffer[i++]) * (1 << 8);
    address += getNibble(serialBuffer[i++]) * (1 << 4);
    address += getNibble(serialBuffer[i++]);
    return address;
}

byte get8BitValue(byte index) {
    byte i = index;
    byte data;
    data  = getNibble(serialBuffer[i++]) * (1 << 4);
    data += getNibble(serialBuffer[i++]);
    return data;
}

int getNibble(unsigned char myChar) {
    int nibble = myChar;
    if (nibble > 'F') nibble -= 0x20;  // lower to upper case
    nibble -= '0';
    if (nibble > 9) nibble -= 7; // offset 9+1 - A
    return nibble;
}

void viewPorts() { // A, B, C, L, c
    bool repeat = 0;
    if (serialBuffer[1] == '+') repeat = 1;
    byte oldValA = 0;
    byte valueA;
    byte oldValB = 0;
    byte valueB;
    byte oldValC = 0;
    byte valueC;
    byte oldValL = 0;
    byte valueL;
    byte oldValc = 0;
    byte valuec;
    PORTA = 0xFF; // pull up
    DDRA = PORT_IN;
    PORTB = 0xFF; // pull up
    DDRB = PORT_IN;
    PORTC = 0xFF; // pull up
    DDRC = PORT_IN;
    PORTL = 0xFF; // pull up
    DDRL = PORT_IN;
    pinMode(MCP20, INPUT_PULLUP);
    pinMode(MCP21, INPUT_PULLUP);
    pinMode(MCP22, INPUT_PULLUP);
    pinMode(MCP23, INPUT_PULLUP);
    pinMode(MCP24, INPUT_PULLUP);
    do {
        valueA = PINA;
        if (oldValA != valueA) {
            Serial.print("A");
            if (repeat) Serial.print("+");
            Serial.print(": ");
            printBin(valueA);
            Serial.println();
            oldValA = valueA;
        }
        valueB  = digitalRead(MCB0);
        valueB += digitalRead(MCE);
        valueB += digitalRead(MCNMI);
        valueB += digitalRead(MCIRQ);
        valueB += digitalRead(MCG0);
        valueB += digitalRead(MCRW);
        valueB += digitalRead(MCAS);
        valueB += digitalRead(MCRES);
        if (oldValB != valueB) {
            Serial.print("B");
            if (repeat) Serial.print("+");
            Serial.print(": ");
            printBin(valueB);
            Serial.println();
            oldValB = valueB;
        }
        valueC = PINC;
        if (oldValC != valueC) {
            Serial.print("C");
            if (repeat) Serial.print("+");
            Serial.print(": ");
            printBin(valueC);
            Serial.println();
            oldValC = valueC;
        }
        valueL = PINL;
        if (oldValL != valueL) {
            Serial.print("L");
            if (repeat) Serial.print("+");
            Serial.print(": ");
            printBin(valueL);
            Serial.println();
            oldValL = valueL;
        }
        valuec  = digitalRead(MCP20) * 0x01;
        valuec += digitalRead(MCP21) * 0x02;
        valuec += digitalRead(MCP22) * 0x04;
        valuec += digitalRead(MCP23) * 0x08;
        valuec += digitalRead(MCP24) * 0x10;
        if (oldValc != valuec) {
            Serial.print("c");
            if (repeat) Serial.print("+");
            Serial.print(": ");
            printBin(valuec);
            Serial.println();
            oldValc = valuec;
        }      
        if (stopIt()) repeat = 0;
    } while(repeat);
}
